/**
 * Copyright Notice
 *
 * This is a work of the U.S. Government and is not subject to copyright
 * protection in the United States. Foreign copyrights may apply.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package gov.vha.isaac.utils.file_transfer;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.channels.ClosedByInterruptException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.NodeList;

import gov.vha.isaac.utils.file_transfer.FileTransfer.FileActionResult;

/**
 * 
 * {@link FileTransferUtils}
 *
 * @author <a href="mailto:joel.kniaz.list@gmail.com">Joel Kniaz</a>
 *
 */
public class FileTransferUtils {
	static class Paths {
		private final String remote;
		private final String local;
	
		/**
		 * @param remote
		 * @param local
		 */
		public Paths(String remote, String local) {
			super();
			this.remote = remote;
			this.local = local;
		}
	
		/**
		 * @return the remote
		 */
		public String getRemote() {
			return remote;
		}
	
		/**
		 * @return the local
		 */
		public String getLocal() {
			return local;
		}
	}

	private static Logger log = LoggerFactory.getLogger(FileTransferUtils.class);

	private FileTransferUtils() {}
	
	static String basename(String path) {
		return path.substring(path.lastIndexOf('/') + 1, path.length());
	}
	
	static String getDirectoryListingHtmlPage(String directoryUri, String username, String password) throws IOException {
		URL url = new URL(directoryUri);
		HttpURLConnection con = NetworkUtils.getConnection(url, username, password); // url.openConnection();
		con.connect();

		java.io.BufferedReader in = new java.io.BufferedReader
				(new java.io.InputStreamReader(con.getInputStream()));
		
		StringBuilder sb = new StringBuilder();
		String line = null;
		for (; (line = in.readLine()) != null; ) {
			if (! line.trim().startsWith("<link ")
					&& ! line.trim().equals("&nbsp;")) {
				// Web pages sometimes have <link> tags without terminators
				// and &nbsp;, which the parser can't handle
				sb.append(line + "\n");
			}
		}
		
		return sb.toString();
	}

	static Set<String> getFileNamesFromDirectoryListingHtmlPage(String html) {
		Set<String> fileNames = new HashSet<>();

		NodeList nodeList = XMLUtils.getNodeList(html, "/html/body/table/tr/td/a");

		for (int i = 0; i < nodeList.getLength(); ++i) {
			String fileOrDirName = nodeList.item(i).getTextContent();
			if (! fileOrDirName.trim().endsWith("/")
					&& ! fileOrDirName.trim().equals("Parent Directory")
					&& ! fileOrDirName.trim().endsWith(".md5")
					&& ! fileOrDirName.trim().endsWith(".sha1")) {
				fileNames.add(fileOrDirName);
			}
		}
		
		return Collections.unmodifiableSet(fileNames);
	}

	
	static boolean validateChecksum(File file, File sha1File) {
		try {
			String expectedSha1Value = java.nio.file.Files.readAllLines(sha1File.toPath()).get(0);
			String calculatedSha1Value = ChecksumGenerator.calculateChecksum("SHA1", file);
			if (calculatedSha1Value != null && !expectedSha1Value.equals(calculatedSha1Value)) {
				return false;
			} else {
				return true;
			}
		} catch (Exception e) {
			if (e instanceof ClosedByInterruptException) {
				log.warn("Failed calculating SHA1 checksum for " + file.getAbsolutePath());
			} else {
				log.warn("Failed calculating SHA1 checksum for " + file.getAbsolutePath(), e);
			}

			return false;
		}
	}
	
	static boolean downloadChecksumFilesAndValidateChecksum(File file, URL remoteFileUrl, String username, String password) {
		return downloadChecksumFilesAndValidateChecksum(file, remoteFileUrl, username, password, true);
	}

	static boolean downloadChecksumFilesAndValidateChecksum(File file, URL remoteFileUrl, String username, String password, boolean keepChecksumFiles) {
		try {
			URL sha1Url = new URL(remoteFileUrl.toString() + ".sha1");
			URL md5Url = new URL(remoteFileUrl.toString() + ".md5");

			boolean sha1FileExists = remoteFileExists(sha1Url, username, password);
			boolean md5FileExists = remoteFileExists(md5Url, username, password);
			
			if (! sha1FileExists && ! md5FileExists) {
				return true;
			}
			
			File checksumFileDir = keepChecksumFiles ? file.getParentFile() : java.nio.file.Files.createTempDirectory("ISAAC").toFile();
			if (! keepChecksumFiles) {
				checksumFileDir.deleteOnExit();
			}

			File sha1File = null;
			if (sha1FileExists) {
				sha1File = download(sha1Url, username, password, checksumFileDir);
				if (! keepChecksumFiles) {
					sha1File.deleteOnExit();
				}
			}
			
			File md5File = null;
			if (md5FileExists) {
				md5File = download(md5Url, username, password, checksumFileDir);
				if (! keepChecksumFiles) {
					md5File.deleteOnExit();
				}
			}

			if (! file.exists()) {
				return false;
			}
			
			boolean validationSucceeded = false;
			
			// If file exists and SHA1 file exists and file exists validate checksum
			if (sha1FileExists) {
				validationSucceeded = validateChecksum(file, sha1File);
			} else {
				// If file exists and SHA1 file DOES NOT exist then automatically pass validation
				validationSucceeded = true;
			}

			if (! keepChecksumFiles) {
				if (sha1File != null) {
					sha1File.delete();
				}
				if (md5File != null) {
					md5File.delete();
				}
				checksumFileDir.delete();
			}
			
			return validationSucceeded;
		} catch (Exception e) {
			log.warn("Failed validating SHA1 checksum for " + file.getAbsolutePath(), e);

			return false;
		}
	}

	static boolean remoteFileExists(URL URLName, String username, String password) {
		try {
			HttpURLConnection con = NetworkUtils.getConnection(URLName, username, password);
			HttpURLConnection.setFollowRedirects(false);
			con.setInstanceFollowRedirects(false);
			con.setRequestMethod("HEAD");
			return (con.getResponseCode() == HttpURLConnection.HTTP_OK);
		}
		catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}  

	static Set<String> listFilesInDirectory(String directoryUri, String username, String password) throws IOException {
		Set<String> fileNames = new HashSet<>();
		if (directoryUri.contains(":")) {
			return getFileNamesFromDirectoryListingHtmlPage(getDirectoryListingHtmlPage(directoryUri, username, password));
		} else {
			File directory = new File(directoryUri);
			if (directory.exists() && directory.canRead() && directory.canExecute()) {
				for (File file : directory.listFiles()) {
					if (! file.isDirectory()) {
						fileNames.add(file.getName());
					}
				}
			}
		}
		
		return Collections.unmodifiableSet(fileNames);
	}
	
	/**
	 * @param url remote file url
	 * @param username login username
	 * @param password login password
	 * @param targetDirectory target directory
	 * @return
	 * @throws Exception
	 * 
	 * This method does not offer any way to cancel download
	 */
	static File download(URL url, String username, String password, File targetDirectory) throws Exception {
		return download(url, username, password, targetDirectory, true);
	}

	/**
	 * @param url remote file url
	 * @param username login username
	 * @param password login password
	 * @param targetDirectory target directory
	 * @param useFileResultsRegistry boolean specifying whether to use static results registry
	 * @return
	 * @throws Exception
	 * 
	 * This method does not offer any way to cancel download
	 */
	static File download(URL url, String username, String password, File targetDirectory, boolean useFileResultsRegistry) throws Exception
	{
		HttpURLConnection httpCon = null;
		FileOutputStream fos = null;
		InputStream in = null;
		
		String fileName = url.toString();
		fileName = fileName.substring(fileName.lastIndexOf('/') + 1, fileName.length());
		
		File file = new File(targetDirectory, fileName);

		if (useFileResultsRegistry && FileTransfer.hasBeenSuccessfullyHandled(file.getAbsoluteFile().toString())) {
			return file;
		}

		if (useFileResultsRegistry) {
			FileTransfer.setFileActionResult(file, FileActionResult.FAILED);
		}
		
		log.debug("Beginning download from " + url);
		if (! targetDirectory.exists()) {
			targetDirectory.mkdirs();
		}
		try {
			httpCon = NetworkUtils.getConnection(url, username, password);
			httpCon.setDoInput(true);
			httpCon.setRequestMethod("GET");
			httpCon.setConnectTimeout(30 * 1000);
			httpCon.setReadTimeout(60 * 60 * 1000);
			in = httpCon.getInputStream();

			int bytesReadSinceShowingProgress = 0;
			byte[] buf = new byte[1048576];
			fos = new FileOutputStream(file);
			int read = 0;
			while ((read = in.read(buf, 0, buf.length)) > 0)
			{
				fos.write(buf, 0, read);
				
				bytesReadSinceShowingProgress += buf.length;
				if (bytesReadSinceShowingProgress >= FileTransfer.IO_SHOW_PROGRESS_BYTES_THRESHOLD) {
					ConsoleUtil.showProgress(); // Show progress for buffered I/O
					bytesReadSinceShowingProgress = 0;
				}

				if (FileTransfer.DEBUG_FAIL_PROCESSING_FILE_NAME != null && fileName.equals(FileTransfer.DEBUG_FAIL_PROCESSING_FILE_NAME)) {
					throw new IOException("intentionally failing download of \"" + fileName + "\" for debugging purposes");
				}
			}
			fos.flush();

			if (useFileResultsRegistry) {
				FileTransfer.setFileActionResult(file, FileActionResult.SUCCEEDED);
			}
			log.debug("Download complete from " + url);

			return file;
		} catch (Exception e) {
			// Register file as unsuccessfully handled
			FileTransfer.setFileActionResult(file, FileActionResult.FAILED);

			log.error("Failed writing to file " + file.getAbsolutePath(), e);
			file.delete();
			throw e;
		} finally {
			if (httpCon != null) {
				httpCon.disconnect();
			}
			if (fos != null) {
				fos.close();
			}
			if (in != null) {
				in.close();
			}
		}
	}

	static String toString(Collection<?> objs) {
		String[] names = new String[objs.size()];
		int index = 0;
		for (Object obj : objs) {
			if (obj == null) {
				names[index++] = null;
			} else if (obj instanceof File) {
				names[index++] = ((File)obj).getPath();
			} else if (obj instanceof URL) {
				names[index++] = ((URL)obj).getPath();
			} else {
				names[index++] = obj.toString();
			}
		}
		
		return Arrays.toString(names);
	}

	public static void main(String...argv) throws Exception {
		//String uri = "http://URL.DNS:PORT
		String uri = "http://URL.DNS:PORT";

		Set<String> fileNames = null;

//		Set<String> fileNames = listFilesInDirectory(uri, "devtest", "devtest");
//		for (String fileName : fileNames) {
//			System.out.println(fileName + ": type=" + getTypeFromFileName(fileName) + ", classifier=" + getClassifierFromFileName(fileName, "isaac-rest", "1.8"));
//		}
		fileNames = MavenFileUtils.listMostRecentMavenMetadataVersionOfEachFileInDirectory(uri, "devtest", "devtest");
		for (String fileName : fileNames) {
			System.out.println(fileName + " (most recent): type=" + MavenFileUtils.getTypeFromFileName(fileName, "metadata", "3.07") + (MavenFileUtils.getClassifierFromFileName(fileName, "metadata", "3.07") != null ? ", classifier=" + MavenFileUtils.getClassifierFromFileName(fileName, "metadata", "3.07") : ""));
		}
		
		uri = "http://URL.DNS:PORT";
//		fileNames = listFilesInDirectory(uri, "devtest", "devtest");
//		for (String fileName : fileNames) {
//			System.out.println(fileName + ": type=" + getTypeFromFileName(fileName, "metadata", "3.07-SNAPSHOT") + (getClassifierFromFileName(fileName, "metadata", "3.07-SNAPSHOT") != null ? ", classifier=" + getClassifierFromFileName(fileName, "metadata", "3.07-SNAPSHOT") : ""));
//		}
		//uri = "http://URL.DNS:PORT";
		fileNames = MavenFileUtils.listMostRecentMavenMetadataVersionOfEachFileInDirectory(uri, "devtest", "devtest");
		for (String fileName : fileNames) {
			System.out.println(fileName + " (most recent): type=" + MavenFileUtils.getTypeFromFileName(fileName, "metadata", "3.07-SNAPSHOT") + (MavenFileUtils.getClassifierFromFileName(fileName, "metadata", "3.07-SNAPSHOT") != null ? ", classifier=" + MavenFileUtils.getClassifierFromFileName(fileName, "metadata", "3.07-SNAPSHOT") : ""));
		}
	}
}
